import { XMLParser } from "fast-xml-parser";
export class OOXML {
    constructor(xml) {
        this.paragraphs = [];
        this.text = "";
        this.origXml = "";
        this.suppressHyphenation = false;
        this.origXml = xml;
        xml.match(/<w:sz w:val="(\d+)" \/>/g)?.forEach(sizeMatch => {
            const size = parseInt(sizeMatch?.match(/<w:sz w:val="(\d+)" \/>/)?.[1] ?? "");
            if (!this.fontSize || size == this.fontSize) {
                this.fontSize = size;
            }
            else {
                throw new Error("Multiple font sizes detected");
            }
        });
        if (xml.match(/<w:suppressAutoHyphens\/>/)) {
            this.suppressHyphenation = true;
        }
        const parser = new XMLParser({ ignoreAttributes: false });
        xml = xml
            .replace(/(<w:commentRange(Start|End) w:id="\d" ?\/>)/g, "<w:r>$1</w:r>") // beware of the dirty hack (xml fast parser does not support a list of heterogenous tags, so make everything <w:r>)
            .replace(/(<w:fldSimple[\s\S]*?>)/g, "<w:r>$1</w:r>");
        this.jObj = parser.parse(xml);
        const bodyNode = this.findNode(this.jObj, node => node["w:body"]);
        if (!bodyNode["w:body"]["w:p"][0]) {
            return;
        }
        this.paragraphs = bodyNode["w:body"]["w:p"].map((p) => {
            if (!p["w:r"]) {
                return [];
            }
            if (!Array.isArray(p["w:r"])) {
                p["w:r"] = [p["w:r"]];
            }
            return p["w:r"].map((chunk, i) => {
                if (chunk["w:t"]) {
                    let pre = "";
                    let post = "";
                    if (p["w:pPr"] && i == 0) {
                        if (p["w:pPr"]["w:pStyle"]) {
                            pre += "::style(" + p["w:pPr"]["w:pStyle"]["@_w:val"] + ")";
                        }
                        if (p["w:pPr"]["w:numPr"]) {
                            pre += "::li(" + p["w:pPr"]["w:numPr"]["w:ilvl"]["@_w:val"] + "," + p["w:pPr"]["w:numPr"]["w:numId"]["@_w:val"] + ")";
                        }
                    }
                    if (chunk["w:rPr"]) {
                        if (chunk["w:rPr"]["w:i"] !== undefined) {
                            pre += "*";
                            post += "*";
                        } // italic 
                        if (chunk["w:rPr"]["w:b"] !== undefined) {
                            pre += "**";
                            post += "**";
                        } // bold 
                    }
                    const inner = (chunk["w:t"]["#text"] ?? chunk["w:t"]) + "";
                    if (inner == "[object Object]") {
                        return "";
                    }
                    return pre + inner + post;
                }
                if (chunk["w:instrText"] && chunk["w:instrText"]["#text"]) {
                    return chunk["w:instrText"]["#text"].replace(/\n/, "").replace(/^/, " ");
                }
                if (chunk["w:fldSimple"]) {
                    return chunk["w:fldSimple"]["@_w:instr"] + "||" + chunk["w:r"]["w:t"]["#text"];
                }
                if (chunk["w:commentRangeStart"]) {
                    return "CMNT(";
                }
                if (chunk["w:commentRangeEnd"]) {
                    return ", " + chunk["w:commentRangeEnd"]["@_w:id"] + ")";
                }
                return "";
            });
        });
        const res = this.paragraphs.flatMap(p => p.map(pp => pp.replace(/\n/g, "")).join("\|\|")
            .replace(/\|\|\s*REF (_Ref\d+)\s*.*?\|\|*(.*?)\|\|/g, "\|\|$1($2)\|\|")) // put ref values inside parenthesis
            .join("\\n\\n")
            .replace(/„?(\|\|)*([^\|]+)(\|\|)*\s?XE "[^"]*"(\\f)?(\|\|)*“?/g, "XE($2)") // put XE values inside parenthesis (ES only)
            .replace(/\|\|/g, " ").replace(/\s+/g, " ")
            .replace(/REF\s*(_Ref\d+)\s*(\.?\d+)/g, "$1($2)") // alt ref syntax
            .replace(/(_Ref(\d+)\([^\)]+\))\s*TC\s*.{0,10}_Ref\2\([^\)]{0,10}\)/g, "XA($1)"); // put Anlagenverzeichnis inside parenthesis (ES only)
        // const res = this.paragraphs.flatMap(p => p.map(pp => pp.replace(/\n/g,"")).join("\|\|")) //   // put ref values inside parenthesis
        //     .join("\\n\\n")
        //     .replace(/(\|\|)?\s*REF\s*(_Ref\d+).{0,4}\|\|\s*.*?([^\|]{0,5}?)(\|\|)/g,"\|\|$2($3)\|\|")
        //     .replace(/„?(\|\|)*([^\|]+)(\|\|)*\s?XE "[^"]*"(\\f)?(\|\|)*“?/g, "XE($2)") // put XE values inside parenthesis (ES only)
        //     .replace(/\|\|/g," ").replace(/\s+/g," ")
        //     .replace(/(_Ref(\d+)\([^\)]+\))\s*TC\s*.{0,10}_Ref\2\([^\)]{0,10}\)/g,"XA($1)") // put Anlagenverzeichnis inside parenthesis (ES only)
        const resWithHeuristicsApplied = res
            .replace(/(_Ref\d+\(.*?\)) ([,.:!;])/g, "$1$2"); // remove space between ref and punctuation
        this.text = resWithHeuristicsApplied;
    }
    getContents() {
        return this.text;
    }
    setContents(text) {
        this.text = text;
    }
    getXml() {
        const suppressAutoHyphens = this.suppressHyphenation ? "<w:suppressAutoHyphens/>" : "";
        let reconstructedParagraphs = this.text.split("\\n\\n").filter(Boolean); // this.paragraphs
        let xmlChunks = ("<w:p>" + reconstructedParagraphs.map(p => "<w:r><w:t xml:space=\"preserve\">" + p + "</w:t></w:r>").join("</w:p><w:p>") + "</w:p>")
            .replace(/(Ziffer|Anlage)?\s*?(_Ref\d+)\(([^\)]+)\)(\s)?/g, (fullmatch, prefix, refNo, refStr, trailingSpace) => {
            return `</w:t></w:r>
            ` + (prefix ? `<w:r><w:t xml:space="preserve">${prefix}&#xa0;</w:t></w:r>` : ``) + `
            <w:r><w:fldChar w:fldCharType="separate"/></w:r>
            <w:fldSimple w:instr=" REF  ${refNo} \\n ">
            <w:r><w:t xml:space="preserve">${refStr + (trailingSpace ? ' ' : '')}</w:t></w:r>
            </w:fldSimple>
            <w:r><w:t xml:space=\"preserve\">`;
        })
            .replace(/\*\*(.*?)\*\*/g, "</w:t></w:r><w:r><w:rPr><w:b /></w:rPr><w:t>$1</w:t></w:r><w:r><w:t>") // convert bold
            .replace(/\*(.*?)\*/g, "</w:t></w:r><w:r><w:rPr><w:i /></w:rPr><w:t>$1</w:t></w:r><w:r><w:t>") // convert italics
            .replace(/<w:r>(<w:t[^>]*>)\s?::style\((.*?)\)/g, `<w:pPr><w:pStyle w:val="$2" />${suppressAutoHyphens}</w:pPr><w:r>$1`) // convert styles
            // .replace(/(.{4})(<w:r><w:t[^>]*>)/g, (m0, m1, m2) => {  // add font size
            //     if(this.fontSize){ 
            //         if(m1=="pPr>"){throw("can't have font size and styles applied at the same time")} // do not add font size if it is already there
            //         return  m1+`<w:rPr><w:sz w:val="${this.fontSize}" /></w:rPr>`+ m2  
            //     }
            // })
            .replace(/::li\((\d+)(,(\d+))?\)/g, `<w:numPr><w:ilvl w:val="$1" /><w:numId w:val="$3" /></w:numPr>`) // convert lists
            .replace(/XE\(([^\)]+)\)/g, `
            „</w:t></w:r>
            <w:r w:rsidRPr="00E62F86"><w:rPr><w:rFonts w:asciiTheme="majorHAnsi" w:hAnsiTheme="majorHAnsi" /></w:rPr><w:t>$1</w:t></w:r>
            <w:r><w:rPr><w:bCs /></w:rPr><w:fldChar w:fldCharType="begin" /></w:r>
            <w:r><w:rPr><w:bCs /></w:rPr><w:instrText xml:space="preserve"> XE "$1" </w:instrText></w:r>
            <w:r><w:rPr><w:bCs /></w:rPr><w:fldChar w:fldCharType="end" /></w:r>
            <w:r><w:t xml:space=\"preserve\">“`) // convert definitionen
            .replace(/XA\(([^\)]+)\)/, `
            <w:r w:rsidRPr="00BA2B92"><w:rPr><w:rFonts w:asciiTheme="majorHAnsi" w:hAnsiTheme="majorHAnsi" w:cstheme="majorHAnsi" /><w:bCs /></w:rPr><w:t> </w:t></w:r>
            <w:r w:rsidRPr="00BA2B92"><w:rPr><w:rFonts w:asciiTheme="majorHAnsi" w:hAnsiTheme="majorHAnsi" w:cstheme="majorHAnsi" /><w:bCs /> </w:rPr><w:fldChar w:fldCharType="begin" /></w:r>
            <w:r w:rsidRPr="00BA2B92"><w:rPr><w:rFonts w:asciiTheme="majorHAnsi" w:hAnsiTheme="majorHAnsi" w:cstheme="majorHAnsi" /><w:bCs /></w:rPr><w:instrText xml:space="preserve"> REF _Ref133327951 \r \h </w:instrText></w:r>
            <w:r w:rsidRPr="00BA2B92"><w:rPr><w:rFonts w:asciiTheme="majorHAnsi" w:hAnsiTheme="majorHAnsi" w:cstheme="majorHAnsi" /><w:bCs /></w:rPr></w:r>
            <w:r w:rsidRPr="00BA2B92"><w:rPr><w:rFonts w:asciiTheme="majorHAnsi" w:hAnsiTheme="majorHAnsi" w:cstheme="majorHAnsi" /><w:bCs /></w:rPr><w:fldChar w:fldCharType="separate" /></w:r>
            <w:r><w:rPr><w:rFonts w:asciiTheme="majorHAnsi" w:hAnsiTheme="majorHAnsi" w:cstheme="majorHAnsi" /><w:bCs /></w:rPr><w:t>12.2.3</w:t></w:r>
            <w:r w:rsidRPr="00BA2B92"><w:rPr><w:rFonts w:asciiTheme="majorHAnsi" w:hAnsiTheme="majorHAnsi" w:cstheme="majorHAnsi" /><w:bCs /></w:rPr><w:fldChar w:fldCharType="end" /></w:r>
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:fldChar w:fldCharType="begin" />
            </w:r>
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:instrText xml:space="preserve"> TC "</w:instrText>
            </w:r>
            <w:r w:rsidRPr="00BA2B92">
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:instrText xml:space="preserve"> </w:instrText>
            </w:r>
            <w:bookmarkStart w:id="1" w:name="_Toc133330445" />
            <w:r w:rsidRPr="00BA2B92">
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:instrText>Anlage</w:instrText>
            </w:r>
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:instrText> </w:instrText>
            </w:r>
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:fldChar w:fldCharType="begin" />
            </w:r>
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:instrText xml:space="preserve"> REF _Ref133327951 \r \h </w:instrText>
            </w:r>
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
            </w:r>
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:fldChar w:fldCharType="separate" />
            </w:r>
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:instrText>12.2.3</w:instrText>
            </w:r>
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:fldChar w:fldCharType="end" />
            </w:r>
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:instrText>: Verkäufergesellschaften</w:instrText>
            </w:r>
            <w:bookmarkEnd w:id="1" />
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:instrText xml:space="preserve">"\f C\I "1" </w:instrText>
            </w:r>
            <w:r>
                <w:rPr>
                    <w:bCs />
                </w:rPr>
                <w:fldChar w:fldCharType="end" />
            </w:r>`) // convert anlagenverzeichnis
            .replace(/CMNT\((.*?), (\d+)\)/, `
            </w:t></w:r>
            <w:commentRangeStart w:id="$2" />
            <w:r><w:t xml:space="preserve">$1 </w:t></w:r>
            <w:commentRangeEnd w:id="$2" />
            <w:r><w:rPr><w:rStyle w:val="Kommentarzeichen" /></w:rPr><w:commentReference w:id="$2" /></w:r>
            <w:r><w:t xml:space=\"preserve\">`) // convert comments
            .replace(/<w:r>\n?\s*<w:t>\s*<\/w:t>\n?\s*<\/w:r>/g, "")
            .replace(/<w:p><w:r><\/w:r><\/w:p>/g, "")
            .split("\n")
            .map(s => s.trim())
            .filter(s => s)
            .join("\n");
        const res = "<pkg:package xmlns:pkg='http://schemas.microsoft.com/office/2006/xmlPackage'><pkg:part pkg:name='/_rels/.rels' pkg:contentType='application/vnd.openxmlformats-package.relationships+xml' pkg:padding='512'><pkg:xmlData><Relationships xmlns='http://schemas.openxmlformats.org/package/2006/relationships'><Relationship Id='rId1' Type='http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument' Target='word/document.xml'/></Relationships></pkg:xmlData></pkg:part><pkg:part pkg:name='/word/document.xml' pkg:contentType='application/vnd.openxmlformats-officedocument.wordprocessingml.document.main+xml'><pkg:xmlData><w:document xmlns:w='http://schemas.openxmlformats.org/wordprocessingml/2006/main' ><w:body>" +
            xmlChunks +
            "</w:body></w:document></pkg:xmlData></pkg:part></pkg:package>";
        //return this.origXml.split('<pkg:part pkg:name="/word/comments.xml"')[0]//+ "</pkg:package>"
        return res;
    }
    findNode(nodes, testFn) {
        if (Array.isArray(nodes)) {
            for (const node of nodes) {
                if (testFn(node)) {
                    return node;
                }
                const found = this.findNode(node, testFn);
                if (found) {
                    return found;
                }
            }
        }
        else if (typeof nodes === "object") {
            for (const key in nodes) {
                if (Object.prototype.hasOwnProperty.call(nodes, key)) {
                    const node = nodes[key];
                    if (testFn(node)) {
                        return node;
                    }
                    const found = this.findNode(node, testFn);
                    if (found) {
                        return found;
                    }
                }
            }
        }
        return testFn(nodes) ? nodes : null;
    }
}
